# Server Configuration Guide
## Monitoring System Requirements

### 🖥️ **Web Server Requirements**

#### **Minimum Requirements:**
- **Web Server**: Apache 2.4+ or Nginx 1.18+
- **PHP Version**: PHP 7.4+ (Recommended: PHP 8.1+)
- **Database**: MySQL 5.7+ or MariaDB 10.3+
- **SSL Certificate**: Required for HTTPS (recommended)
- **Disk Space**: 1GB+ (for media files)
- **RAM**: 512MB+ (1GB+ recommended)

#### **Recommended Configuration:**
- **Web Server**: Apache 2.4 with mod_rewrite enabled
- **PHP Version**: PHP 8.1 or PHP 8.2
- **Database**: MySQL 8.0+ or MariaDB 10.6+
- **SSL Certificate**: Let's Encrypt or commercial SSL
- **Disk Space**: 5GB+ (for media files and logs)
- **RAM**: 2GB+ (for better performance)

---

### 📋 **PHP Configuration**

#### **Required PHP Extensions:**
```ini
; Core extensions
extension=pdo
extension=pdo_mysql
extension=json
extension=curl
extension=fileinfo
extension=openssl
extension=mbstring
extension=zip

; Optional but recommended
extension=gd
extension=imagick
extension=redis
extension=memcached
```

#### **PHP Settings (php.ini):**
```ini
; File upload settings
upload_max_filesize = 50M
post_max_size = 50M
max_file_uploads = 20
max_execution_time = 300
max_input_time = 300
memory_limit = 256M

; Session settings
session.gc_maxlifetime = 3600
session.cookie_lifetime = 3600
session.use_strict_mode = 1

; Security settings
expose_php = Off
allow_url_fopen = Off
allow_url_include = Off
disable_functions = exec,passthru,shell_exec,system,proc_open,popen

; Error reporting (for production)
display_errors = Off
log_errors = On
error_log = /var/log/php_errors.log
```

---

### 🗄️ **Database Configuration**

#### **MySQL/MariaDB Settings:**
```sql
-- Create database
CREATE DATABASE swaminar_mobiletr 
CHARACTER SET utf8mb4 
COLLATE utf8mb4_unicode_ci;

-- Create user
CREATE USER 'swaminar_mobiletr'@'localhost' 
IDENTIFIED BY 'Ashish123@';

-- Grant permissions
GRANT SELECT, INSERT, UPDATE, DELETE, CREATE, DROP, INDEX, ALTER 
ON swaminar_mobiletr.* 
TO 'swaminar_mobiletr'@'localhost';

-- Flush privileges
FLUSH PRIVILEGES;
```

#### **Database Optimization:**
```sql
-- Optimize for monitoring system
SET GLOBAL innodb_buffer_pool_size = 1G;
SET GLOBAL max_connections = 200;
SET GLOBAL query_cache_size = 64M;
SET GLOBAL tmp_table_size = 64M;
SET GLOBAL max_heap_table_size = 64M;
```

---

### 🔧 **Apache Configuration**

#### **Virtual Host Configuration:**
```apache
<VirtualHost *:80>
    ServerName mobiletr.upastithi.in
    DocumentRoot /home/swaminar/mobiletr.upastithi.in
    
    # Redirect HTTP to HTTPS
    RewriteEngine On
    RewriteCond %{HTTPS} off
    RewriteRule ^(.*)$ https://%{HTTP_HOST}%{REQUEST_URI} [L,R=301]
</VirtualHost>

<VirtualHost *:443>
    ServerName mobiletr.upastithi.in
    DocumentRoot /home/swaminar/mobiletr.upastithi.in
    
    # SSL Configuration
    SSLEngine on
    SSLCertificateFile /path/to/certificate.crt
    SSLCertificateKeyFile /path/to/private.key
    
    # Security headers
    Header always set X-Content-Type-Options nosniff
    Header always set X-Frame-Options DENY
    Header always set X-XSS-Protection "1; mode=block"
    Header always set Strict-Transport-Security "max-age=31536000; includeSubDomains"
    
    # PHP configuration
    <Directory /home/swaminar/mobiletr.upastithi.in>
        AllowOverride All
        Require all granted
        
        # Security rules
        <Files "*.php">
            Require all granted
        </Files>
        
        # Block access to sensitive files
        <FilesMatch "\.(htaccess|htpasswd|ini|log|sh|sql|conf)$">
            Require all denied
        </FilesMatch>
    </Directory>
    
    # File upload limits
    LimitRequestBody 52428800  # 50MB
    
    # CORS headers for API
    <LocationMatch "^/php/">
        Header always set Access-Control-Allow-Origin "*"
        Header always set Access-Control-Allow-Methods "GET, POST, OPTIONS"
        Header always set Access-Control-Allow-Headers "Content-Type, X-Device-ID, X-Timestamp, X-Auth-Token, X-Data-Type"
    </LocationMatch>
</VirtualHost>
```

#### **.htaccess Configuration:**
```apache
# Security
Options -Indexes
ServerSignature Off

# Block access to sensitive files
<FilesMatch "\.(htaccess|htpasswd|ini|log|sh|sql|conf)$">
    Order Allow,Deny
    Deny from all
</FilesMatch>

# Block access to config directory
<Directory "config">
    Order Allow,Deny
    Deny from all
</Directory>

# Enable compression
<IfModule mod_deflate.c>
    AddOutputFilterByType DEFLATE text/plain
    AddOutputFilterByType DEFLATE text/html
    AddOutputFilterByType DEFLATE text/xml
    AddOutputFilterByType DEFLATE text/css
    AddOutputFilterByType DEFLATE application/xml
    AddOutputFilterByType DEFLATE application/xhtml+xml
    AddOutputFilterByType DEFLATE application/rss+xml
    AddOutputFilterByType DEFLATE application/javascript
    AddOutputFilterByType DEFLATE application/x-javascript
</IfModule>

# Cache control
<IfModule mod_expires.c>
    ExpiresActive On
    ExpiresByType text/css "access plus 1 month"
    ExpiresByType application/javascript "access plus 1 month"
    ExpiresByType image/png "access plus 1 month"
    ExpiresByType image/jpg "access plus 1 month"
    ExpiresByType image/jpeg "access plus 1 month"
    ExpiresByType image/gif "access plus 1 month"
</IfModule>
```

---

### 🔒 **Security Configuration**

#### **File Permissions:**
```bash
# Set proper permissions
chmod 755 /home/swaminar/mobiletr.upastithi.in/
chmod 644 /home/swaminar/mobiletr.upastithi.in/php/config/database.php
chmod 755 /home/swaminar/mobiletr.upastithi.in/uploads/
chmod 644 /home/swaminar/mobiletr.upastithi.in/php/*.php
chmod 644 /home/swaminar/mobiletr.upastithi.in/dashboard/*.html
chmod 644 /home/swaminar/mobiletr.upastithi.in/dashboard/*.css
chmod 644 /home/swaminar/mobiletr.upastithi.in/dashboard/*.js

# Make uploads directory writable
chmod 777 /home/swaminar/mobiletr.upastithi.in/uploads/
```

#### **Firewall Rules:**
```bash
# Allow HTTP and HTTPS
ufw allow 80/tcp
ufw allow 443/tcp

# Allow SSH (if needed)
ufw allow 22/tcp

# Allow MySQL (if needed for remote access)
ufw allow 3306/tcp
```

---

### 📁 **Directory Structure**
```
/home/swaminar/mobiletr.upastithi.in/
├── php/
│   ├── config/
│   │   └── database.php          # Database configuration
│   ├── upload.php                # File upload handler
│   ├── api.php                   # API endpoint
│   ├── auth.php                  # Authentication
│   ├── public_api.php            # Public API (no auth)
│   └── get_commands.php          # Command retrieval
├── dashboard/
│   ├── index.html                # Main dashboard
│   ├── styles.css                # Dashboard styles
│   ├── script.js                 # Dashboard JavaScript
│   └── dashboard_fix.html        # Fixed dashboard
├── uploads/                      # Device uploads (writable)
│   └── (device folders)
└── logs/                         # Application logs (writable)
    └── (log files)
```

---

### 🚀 **Performance Optimization**

#### **PHP OPcache (Recommended):**
```ini
; Enable OPcache
opcache.enable=1
opcache.memory_consumption=128
opcache.interned_strings_buffer=8
opcache.max_accelerated_files=4000
opcache.revalidate_freq=2
opcache.fast_shutdown=1
```

#### **MySQL Optimization:**
```sql
-- Add indexes for better performance
ALTER TABLE devices ADD INDEX idx_last_seen (last_seen);
ALTER TABLE call_logs ADD INDEX idx_device_timestamp (device_id, timestamp);
ALTER TABLE locations ADD INDEX idx_device_timestamp (device_id, timestamp);
ALTER TABLE media_files ADD INDEX idx_device_type (device_id, file_type);
```

---

### 📊 **Monitoring & Logging**

#### **Log Files to Monitor:**
```bash
# Web server logs
/var/log/apache2/access.log
/var/log/apache2/error.log

# PHP logs
/var/log/php_errors.log

# MySQL logs
/var/log/mysql/error.log
/var/log/mysql/slow.log

# Application logs
/home/swaminar/mobiletr.upastithi.in/logs/
```

#### **Log Rotation:**
```bash
# Add to /etc/logrotate.d/monitoring
/home/swaminar/mobiletr.upastithi.in/logs/*.log {
    daily
    missingok
    rotate 30
    compress
    delaycompress
    notifempty
    create 644 www-data www-data
}
```

---

### 🔧 **Installation Commands**

#### **Ubuntu/Debian:**
```bash
# Update system
sudo apt update && sudo apt upgrade -y

# Install Apache
sudo apt install apache2 -y

# Install PHP 8.1
sudo apt install php8.1 php8.1-cli php8.1-common php8.1-mysql php8.1-zip php8.1-gd php8.1-mbstring php8.1-curl php8.1-xml php8.1-bcmath -y

# Install MySQL
sudo apt install mysql-server -y

# Enable Apache modules
sudo a2enmod rewrite
sudo a2enmod ssl
sudo a2enmod headers

# Restart services
sudo systemctl restart apache2
sudo systemctl restart mysql
```

#### **CentOS/RHEL:**
```bash
# Install Apache
sudo yum install httpd -y

# Install PHP 8.1
sudo yum install php81 php81-php-cli php81-php-common php81-php-mysqlnd php81-php-zip php81-php-gd php81-php-mbstring php81-php-curl php81-php-xml php81-php-bcmath -y

# Install MySQL
sudo yum install mysql-server -y

# Start services
sudo systemctl start httpd
sudo systemctl start mysqld
sudo systemctl enable httpd
sudo systemctl enable mysqld
```

---

### ✅ **Verification Checklist**

- [ ] Web server running (Apache/Nginx)
- [ ] PHP 7.4+ installed with required extensions
- [ ] MySQL/MariaDB running
- [ ] Database created with correct credentials
- [ ] SSL certificate installed (recommended)
- [ ] File permissions set correctly
- [ ] Firewall configured
- [ ] Log rotation configured
- [ ] Monitoring system tested

---

### 🆘 **Troubleshooting**

#### **Common Issues:**
1. **Database Connection Failed**: Check credentials in `database.php`
2. **Permission Denied**: Check file permissions (755 for directories, 644 for files)
3. **Upload Failed**: Check `upload_max_filesize` and `post_max_size` in php.ini
4. **API 405 Error**: Check Apache mod_rewrite is enabled
5. **CORS Issues**: Check Apache headers configuration

#### **Debug Commands:**
```bash
# Check PHP version
php -v

# Check PHP modules
php -m

# Check Apache status
sudo systemctl status apache2

# Check MySQL status
sudo systemctl status mysql

# Check logs
tail -f /var/log/apache2/error.log
tail -f /var/log/php_errors.log
```

---

This configuration will ensure your monitoring system runs smoothly and securely! 🚀
