<?php
// Direct database check - bypass API authentication
echo "🗄️ Direct Database Check\n";
echo "========================\n\n";

require_once 'php/config/database.php';

// Test database connection
echo "1️⃣ Testing Database Connection...\n";
$pdo = getDatabaseConnection();
if (!$pdo) {
    echo "❌ Database connection failed!\n";
    exit(1);
}
echo "✅ Database connection successful!\n";

// Check devices table
echo "\n2️⃣ Checking Devices Table...\n";
try {
    $stmt = $pdo->query("SELECT * FROM devices ORDER BY last_seen DESC");
    $devices = $stmt->fetchAll();
    
    if (empty($devices)) {
        echo "ℹ️ No devices found in database.\n";
        echo "This means either:\n";
        echo "- No Android devices have connected yet\n";
        echo "- Device registration is failing\n";
        echo "- There's an issue with the upload.php endpoint\n";
    } else {
        echo "✅ Found " . count($devices) . " devices in database:\n";
        foreach ($devices as $device) {
            echo "  📱 Device ID: " . $device['device_id'] . "\n";
            echo "     Name: " . $device['device_name'] . "\n";
            echo "     Last Seen: " . $device['last_seen'] . "\n";
            echo "     Created: " . $device['created_at'] . "\n";
            echo "     ---\n";
        }
    }
} catch (Exception $e) {
    echo "❌ Error querying devices: " . $e->getMessage() . "\n";
}

// Check locations table
echo "\n3️⃣ Checking Locations Table...\n";
try {
    $stmt = $pdo->query("SELECT * FROM locations ORDER BY timestamp DESC LIMIT 5");
    $locations = $stmt->fetchAll();
    
    if (empty($locations)) {
        echo "ℹ️ No location data found.\n";
    } else {
        echo "✅ Found " . count($locations) . " recent location entries:\n";
        foreach ($locations as $location) {
            echo "  📍 Device: " . $location['device_id'] . "\n";
            echo "     Location: " . $location['latitude'] . ", " . $location['longitude'] . "\n";
            echo "     Address: " . $location['address'] . "\n";
            echo "     Time: " . date('Y-m-d H:i:s', $location['timestamp'] / 1000) . "\n";
            echo "     ---\n";
        }
    }
} catch (Exception $e) {
    echo "❌ Error querying locations: " . $e->getMessage() . "\n";
}

// Check media files table
echo "\n4️⃣ Checking Media Files Table...\n";
try {
    $stmt = $pdo->query("SELECT * FROM media_files ORDER BY uploaded_at DESC LIMIT 5");
    $media = $stmt->fetchAll();
    
    if (empty($media)) {
        echo "ℹ️ No media files found.\n";
    } else {
        echo "✅ Found " . count($media) . " recent media files:\n";
        foreach ($media as $file) {
            echo "  📁 Device: " . $file['device_id'] . "\n";
            echo "     File: " . $file['file_name'] . "\n";
            echo "     Type: " . $file['file_type'] . "\n";
            echo "     Size: " . number_format($file['file_size']) . " bytes\n";
            echo "     Uploaded: " . $file['uploaded_at'] . "\n";
            echo "     ---\n";
        }
    }
} catch (Exception $e) {
    echo "❌ Error querying media files: " . $e->getMessage() . "\n";
}

// Test device registration manually
echo "\n5️⃣ Testing Manual Device Registration...\n";
try {
    $testDeviceId = "manual_test_" . time();
    $stmt = $pdo->prepare("
        INSERT INTO devices (device_id, device_name) 
        VALUES (?, ?) 
        ON DUPLICATE KEY UPDATE last_seen = CURRENT_TIMESTAMP
    ");
    $result = $stmt->execute([$testDeviceId, "Manual Test Device"]);
    
    if ($result) {
        echo "✅ Manual device registration successful!\n";
        
        // Verify it was added
        $stmt = $pdo->prepare("SELECT * FROM devices WHERE device_id = ?");
        $stmt->execute([$testDeviceId]);
        $device = $stmt->fetch();
        
        if ($device) {
            echo "✅ Device found in database after registration!\n";
            echo "  Device ID: " . $device['device_id'] . "\n";
            echo "  Name: " . $device['device_name'] . "\n";
            echo "  Last Seen: " . $device['last_seen'] . "\n";
        } else {
            echo "❌ Device not found after registration!\n";
        }
        
        // Clean up test device
        $stmt = $pdo->prepare("DELETE FROM devices WHERE device_id = ?");
        $stmt->execute([$testDeviceId]);
        echo "✅ Test device cleaned up.\n";
    } else {
        echo "❌ Manual device registration failed!\n";
    }
} catch (Exception $e) {
    echo "❌ Manual registration error: " . $e->getMessage() . "\n";
}

echo "\n✅ Direct Database Check Completed!\n";
echo "==================================\n";
echo "If you see devices in the database but not in the dashboard,\n";
echo "the issue is with the API authentication or dashboard JavaScript.\n";
echo "If you don't see any devices, the issue is with device registration.\n";
?>

