<?php
require_once 'config/database.php';
require_once 'auth.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, X-Device-ID, X-Timestamp, X-Auth-Token, X-Data-Type');

// Handle preflight OPTIONS requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Check authentication
if (!isAuthenticated()) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Authentication required']);
    exit;
}

// Initialize database
if (!initializeDatabase()) {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Database initialization failed']);
    exit;
}

try {
    $pdo = getDatabaseConnection();
    if (!$pdo) {
        throw new Exception('Database connection failed');
    }
    
    $action = $_GET['action'] ?? '';
    
    switch ($action) {
        case 'devices':
            getDevices($pdo);
            break;
        case 'call_logs':
            getCallLogs($pdo);
            break;
        case 'locations':
            getLocations($pdo);
            break;
        case 'media':
            getMedia($pdo);
            break;
        case 'commands':
            getCommands($pdo);
            break;
        case 'device_stats':
            getDeviceStats($pdo);
            break;
        default:
            throw new Exception('Invalid action');
    }
    
} catch (Exception $e) {
    error_log("API error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}

function getDevices($pdo) {
    $stmt = $pdo->query("
        SELECT 
            d.*,
            COUNT(DISTINCT cl.id) as call_count,
            COUNT(DISTINCT l.id) as location_count,
            COUNT(DISTINCT mf.id) as media_count,
            MAX(cl.timestamp) as last_call,
            MAX(l.timestamp) as last_location,
            MAX(mf.timestamp) as last_media
        FROM devices d
        LEFT JOIN call_logs cl ON d.device_id = cl.device_id
        LEFT JOIN locations l ON d.device_id = l.device_id
        LEFT JOIN media_files mf ON d.device_id = mf.device_id
        GROUP BY d.device_id
        ORDER BY d.last_seen DESC
    ");
    
    $devices = $stmt->fetchAll();
    echo json_encode(['success' => true, 'data' => $devices]);
}

function getCallLogs($pdo) {
    $deviceId = $_GET['device_id'] ?? '';
    $limit = min((int)($_GET['limit'] ?? 100), 1000);
    $offset = max((int)($_GET['offset'] ?? 0), 0);
    
    $sql = "SELECT * FROM call_logs";
    $params = [];
    
    if ($deviceId) {
        $sql .= " WHERE device_id = ?";
        $params[] = $deviceId;
    }
    
    $sql .= " ORDER BY timestamp DESC LIMIT ? OFFSET ?";
    $params[] = $limit;
    $params[] = $offset;
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    
    $callLogs = $stmt->fetchAll();
    echo json_encode(['success' => true, 'data' => $callLogs]);
}

function getLocations($pdo) {
    $deviceId = $_GET['device_id'] ?? '';
    $limit = min((int)($_GET['limit'] ?? 100), 1000);
    $offset = max((int)($_GET['offset'] ?? 0), 0);
    
    $sql = "SELECT * FROM locations";
    $params = [];
    
    if ($deviceId) {
        $sql .= " WHERE device_id = ?";
        $params[] = $deviceId;
    }
    
    $sql .= " ORDER BY timestamp DESC LIMIT ? OFFSET ?";
    $params[] = $limit;
    $params[] = $offset;
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    
    $locations = $stmt->fetchAll();
    echo json_encode(['success' => true, 'data' => $locations]);
}

function getMedia($pdo) {
    $deviceId = $_GET['device_id'] ?? '';
    $type = $_GET['type'] ?? '';
    $limit = min((int)($_GET['limit'] ?? 50), 500);
    $offset = max((int)($_GET['offset'] ?? 0), 0);
    
    $sql = "SELECT * FROM media_files";
    $params = [];
    $conditions = [];
    
    if ($deviceId) {
        $conditions[] = "device_id = ?";
        $params[] = $deviceId;
    }
    
    if ($type) {
        $conditions[] = "file_type = ?";
        $params[] = $type;
    }
    
    if (!empty($conditions)) {
        $sql .= " WHERE " . implode(" AND ", $conditions);
    }
    
    $sql .= " ORDER BY timestamp DESC LIMIT ? OFFSET ?";
    $params[] = $limit;
    $params[] = $offset;
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    
    $media = $stmt->fetchAll();
    echo json_encode(['success' => true, 'data' => $media]);
}

function getCommands($pdo) {
    $deviceId = $_GET['device_id'] ?? '';
    $limit = min((int)($_GET['limit'] ?? 50), 500);
    $offset = max((int)($_GET['offset'] ?? 0), 0);
    
    $sql = "SELECT * FROM remote_commands";
    $params = [];
    
    if ($deviceId) {
        $sql .= " WHERE device_id = ?";
        $params[] = $deviceId;
    }
    
    $sql .= " ORDER BY created_at DESC LIMIT ? OFFSET ?";
    $params[] = $limit;
    $params[] = $offset;
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    
    $commands = $stmt->fetchAll();
    echo json_encode(['success' => true, 'data' => $commands]);
}

function getDeviceStats($pdo) {
    $deviceId = $_GET['device_id'] ?? '';
    
    if (!$deviceId) {
        throw new Exception('Device ID required');
    }
    
    // Get call statistics
    $callStats = $pdo->prepare("
        SELECT 
            type,
            COUNT(*) as count,
            AVG(duration) as avg_duration
        FROM call_logs 
        WHERE device_id = ? 
        GROUP BY type
    ");
    $callStats->execute([$deviceId]);
    $callStatsData = $callStats->fetchAll();
    
    // Get location statistics
    $locationStats = $pdo->prepare("
        SELECT 
            COUNT(*) as total_locations,
            MIN(latitude) as min_lat,
            MAX(latitude) as max_lat,
            MIN(longitude) as min_lng,
            MAX(longitude) as max_lng
        FROM locations 
        WHERE device_id = ?
    ");
    $locationStats->execute([$deviceId]);
    $locationStatsData = $locationStats->fetch();
    
    // Get media statistics
    $mediaStats = $pdo->prepare("
        SELECT 
            file_type,
            COUNT(*) as count,
            SUM(file_size) as total_size
        FROM media_files 
        WHERE device_id = ? 
        GROUP BY file_type
    ");
    $mediaStats->execute([$deviceId]);
    $mediaStatsData = $mediaStats->fetchAll();
    
    // Get recent activity
    $recentActivity = $pdo->prepare("
        SELECT 'call' as type, timestamp, 'Call log entry' as description
        FROM call_logs WHERE device_id = ?
        UNION ALL
        SELECT 'location' as type, timestamp, CONCAT('Location: ', latitude, ', ', longitude) as description
        FROM locations WHERE device_id = ?
        UNION ALL
        SELECT 'media' as type, timestamp, CONCAT('Media: ', file_name) as description
        FROM media_files WHERE device_id = ?
        ORDER BY timestamp DESC
        LIMIT 20
    ");
    $recentActivity->execute([$deviceId, $deviceId, $deviceId]);
    $recentActivityData = $recentActivity->fetchAll();
    
    echo json_encode([
        'success' => true,
        'data' => [
            'call_stats' => $callStatsData,
            'location_stats' => $locationStatsData,
            'media_stats' => $mediaStatsData,
            'recent_activity' => $recentActivityData
        ]
    ]);
}
?>
