<?php
// Database setup script - run this after finding correct credentials
echo "🗄️ Database Setup Script\n";
echo "========================\n\n";

// Update these with your correct credentials
$DB_HOST = 'localhost';
$DB_NAME = 'monitoring_db';
$DB_USER = 'root';
$DB_PASS = '';

echo "Using credentials:\n";
echo "Host: $DB_HOST\n";
echo "Database: $DB_NAME\n";
echo "User: $DB_USER\n";
echo "Password: " . ($DB_PASS ?: 'empty') . "\n\n";

try {
    // Test connection
    echo "1️⃣ Testing database connection...\n";
    $dsn = "mysql:host=$DB_HOST;dbname=$DB_NAME;charset=utf8mb4";
    $pdo = new PDO($dsn, $DB_USER, $DB_PASS, [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_EMULATE_PREPARES => false,
    ]);
    echo "✅ Database connection successful!\n\n";
    
    // Create tables
    echo "2️⃣ Creating database tables...\n";
    
    // Devices table
    $pdo->exec("CREATE TABLE IF NOT EXISTS devices (
        id INT AUTO_INCREMENT PRIMARY KEY,
        device_id VARCHAR(255) UNIQUE NOT NULL,
        device_name VARCHAR(255),
        last_seen TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    )");
    echo "✅ Devices table created/verified\n";
    
    // Call logs table
    $pdo->exec("CREATE TABLE IF NOT EXISTS call_logs (
        id INT AUTO_INCREMENT PRIMARY KEY,
        device_id VARCHAR(255) NOT NULL,
        timestamp BIGINT NOT NULL,
        duration BIGINT NOT NULL,
        type VARCHAR(20) NOT NULL,
        number VARCHAR(50) NOT NULL,
        contact_name VARCHAR(255),
        latitude DECIMAL(10, 8),
        longitude DECIMAL(11, 8),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        INDEX idx_device_timestamp (device_id, timestamp)
    )");
    echo "✅ Call logs table created/verified\n";
    
    // Locations table
    $pdo->exec("CREATE TABLE IF NOT EXISTS locations (
        id INT AUTO_INCREMENT PRIMARY KEY,
        device_id VARCHAR(255) NOT NULL,
        latitude DECIMAL(10, 8) NOT NULL,
        longitude DECIMAL(11, 8) NOT NULL,
        accuracy FLOAT,
        timestamp BIGINT NOT NULL,
        address TEXT,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        INDEX idx_device_timestamp (device_id, timestamp)
    )");
    echo "✅ Locations table created/verified\n";
    
    // Media files table
    $pdo->exec("CREATE TABLE IF NOT EXISTS media_files (
        id INT AUTO_INCREMENT PRIMARY KEY,
        device_id VARCHAR(255) NOT NULL,
        file_name VARCHAR(255) NOT NULL,
        file_path VARCHAR(500) NOT NULL,
        file_size BIGINT NOT NULL,
        mime_type VARCHAR(100) NOT NULL,
        file_type VARCHAR(20) NOT NULL,
        timestamp BIGINT NOT NULL,
        uploaded_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        INDEX idx_device_type (device_id, file_type),
        INDEX idx_timestamp (timestamp)
    )");
    echo "✅ Media files table created/verified\n";
    
    // Remote commands table
    $pdo->exec("CREATE TABLE IF NOT EXISTS remote_commands (
        id INT AUTO_INCREMENT PRIMARY KEY,
        device_id VARCHAR(255) NOT NULL,
        command_type VARCHAR(50) NOT NULL,
        command_param TEXT,
        status VARCHAR(20) DEFAULT 'pending',
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        executed_at TIMESTAMP NULL,
        INDEX idx_device_status (device_id, status)
    )");
    echo "✅ Remote commands table created/verified\n";
    
    // Test device registration
    echo "\n3️⃣ Testing device registration...\n";
    $testDeviceId = "setup_test_" . time();
    $stmt = $pdo->prepare("
        INSERT INTO devices (device_id, device_name) 
        VALUES (?, ?) 
        ON DUPLICATE KEY UPDATE last_seen = CURRENT_TIMESTAMP
    ");
    $result = $stmt->execute([$testDeviceId, "Setup Test Device"]);
    
    if ($result) {
        echo "✅ Device registration test successful!\n";
        
        // Verify device exists
        $stmt = $pdo->prepare("SELECT * FROM devices WHERE device_id = ?");
        $stmt->execute([$testDeviceId]);
        $device = $stmt->fetch();
        
        if ($device) {
            echo "✅ Device found in database!\n";
            echo "  Device ID: " . $device['device_id'] . "\n";
            echo "  Name: " . $device['device_name'] . "\n";
            echo "  Last Seen: " . $device['last_seen'] . "\n";
        }
        
        // Clean up test device
        $stmt = $pdo->prepare("DELETE FROM devices WHERE device_id = ?");
        $stmt->execute([$testDeviceId]);
        echo "✅ Test device cleaned up.\n";
    } else {
        echo "❌ Device registration test failed!\n";
    }
    
    // Test API endpoint
    echo "\n4️⃣ Testing API endpoint...\n";
    $apiUrl = "http://mobiletr.upastithi.in/php/public_api.php?action=devices";
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $apiUrl);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode === 200) {
        $data = json_decode($response, true);
        if ($data && isset($data['success']) && $data['success']) {
            echo "✅ API endpoint working! Found " . count($data['data']) . " devices.\n";
        } else {
            echo "❌ API endpoint error: " . ($data['message'] ?? 'Unknown error') . "\n";
        }
    } else {
        echo "❌ API endpoint failed (HTTP $httpCode)\n";
    }
    
    echo "\n🎉 Database setup completed successfully!\n";
    echo "========================================\n";
    echo "Your monitoring system is now ready!\n";
    echo "1. Update your database.php file with these credentials\n";
    echo "2. Your Android app should now be able to register devices\n";
    echo "3. Check your dashboard at: http://mobiletr.upastithi.in/dashboard/\n";
    
} catch (PDOException $e) {
    echo "❌ Database error: " . $e->getMessage() . "\n";
    echo "\nPlease check your database credentials and try again.\n";
} catch (Exception $e) {
    echo "❌ Error: " . $e->getMessage() . "\n";
}
?>
