<?php
// End-to-end device registration test
// This simulates what the Android app does when registering a device

require_once 'php/config/database.php';

echo "🔍 Testing Device Registration Flow End-to-End\n";
echo "==============================================\n\n";

// Test 1: Database Connection
echo "1️⃣ Testing Database Connection...\n";
$pdo = getDatabaseConnection();
if (!$pdo) {
    echo "❌ Database connection failed!\n";
    echo "Please check your database configuration.\n";
    exit(1);
}
echo "✅ Database connection successful!\n\n";

// Test 2: Database Initialization
echo "2️⃣ Testing Database Initialization...\n";
if (initializeDatabase()) {
    echo "✅ Database tables initialized successfully!\n\n";
} else {
    echo "❌ Database initialization failed!\n";
    exit(1);
}

// Test 3: Simulate Android App Device ID Generation
echo "3️⃣ Simulating Android Device ID Generation...\n";
$testDeviceId = "test_device_" . time() . "_" . rand(1000, 9999);
echo "Generated test device ID: $testDeviceId\n\n";

// Test 4: Simulate Android App Authentication Headers
echo "4️⃣ Simulating Android App Authentication...\n";
$timestamp = time() * 1000; // Android uses milliseconds
$apiKey = API_KEY;
$encryptionKey = ENCRYPTION_KEY;

// Simulate the same encryption as Android app
function simulateAndroidEncryption($data, $key) {
    $result = '';
    for ($i = 0; $i < strlen($data); $i++) {
        $result .= chr(ord($data[$i]) ^ ord($key[$i % strlen($key)]));
    }
    return $result;
}

$authToken = simulateAndroidEncryption("$testDeviceId:$timestamp:$apiKey", $encryptionKey);
echo "Generated auth token: " . substr($authToken, 0, 20) . "...\n";

// Test 5: Simulate HTTP Request Headers (as Android app would send)
echo "5️⃣ Simulating HTTP Request Headers...\n";
$headers = [
    'X-Device-ID' => $testDeviceId,
    'X-Timestamp' => $timestamp,
    'X-Auth-Token' => $authToken,
    'X-Data-Type' => 'location'
];

echo "Headers:\n";
foreach ($headers as $key => $value) {
    echo "  $key: $value\n";
}

// Test 6: Test Authentication Validation
echo "\n6️⃣ Testing Authentication Validation...\n";
$isValid = validateDeviceAuth($headers);
if ($isValid) {
    echo "✅ Authentication validation successful!\n";
} else {
    echo "❌ Authentication validation failed!\n";
    echo "This means the Android app and server encryption keys don't match!\n";
    exit(1);
}

// Test 7: Simulate Location Data Upload
echo "\n7️⃣ Simulating Location Data Upload...\n";
$locationData = [
    'latitude' => 28.6139,
    'longitude' => 77.2090,
    'accuracy' => 10.5,
    'timestamp' => $timestamp,
    'address' => 'New Delhi, India'
];

$encryptedLocationData = simulateAndroidEncryption(json_encode($locationData), $encryptionKey);
echo "Location data encrypted and ready for upload.\n";

// Test 8: Simulate upload.php Processing
echo "\n8️⃣ Simulating upload.php Processing...\n";

// Simulate the upload.php logic
try {
    // Register device
    $stmt = $pdo->prepare("
        INSERT INTO devices (device_id, device_name) 
        VALUES (?, ?) 
        ON DUPLICATE KEY UPDATE last_seen = CURRENT_TIMESTAMP
    ");
    $result = $stmt->execute([$testDeviceId, "Test Device $testDeviceId"]);
    
    if ($result) {
        echo "✅ Device registered successfully in database!\n";
    } else {
        echo "❌ Device registration failed!\n";
        exit(1);
    }
    
    // Process location data
    $decryptedData = simulateAndroidEncryption($encryptedLocationData, $encryptionKey);
    $location = json_decode($decryptedData, true);
    
    if ($location) {
        $stmt = $pdo->prepare("
            INSERT INTO locations (device_id, latitude, longitude, accuracy, timestamp, address) 
            VALUES (?, ?, ?, ?, ?, ?)
        ");
        $result = $stmt->execute([
            $testDeviceId,
            $location['latitude'],
            $location['longitude'],
            $location['accuracy'],
            $location['timestamp'],
            $location['address']
        ]);
        
        if ($result) {
            echo "✅ Location data uploaded successfully!\n";
        } else {
            echo "❌ Location data upload failed!\n";
        }
    } else {
        echo "❌ Location data decryption failed!\n";
    }
    
} catch (Exception $e) {
    echo "❌ Upload processing failed: " . $e->getMessage() . "\n";
    exit(1);
}

// Test 9: Verify Device in Database
echo "\n9️⃣ Verifying Device in Database...\n";
$stmt = $pdo->prepare("SELECT * FROM devices WHERE device_id = ?");
$stmt->execute([$testDeviceId]);
$device = $stmt->fetch();

if ($device) {
    echo "✅ Device found in database!\n";
    echo "  Device ID: " . $device['device_id'] . "\n";
    echo "  Device Name: " . $device['device_name'] . "\n";
    echo "  Last Seen: " . $device['last_seen'] . "\n";
    echo "  Created: " . $device['created_at'] . "\n";
} else {
    echo "❌ Device not found in database!\n";
    exit(1);
}

// Test 10: Test API Endpoint
echo "\n🔟 Testing API Endpoint...\n";
$apiUrl = "https://mobiletr.upastithi.in/php/api.php?action=devices";
$context = stream_context_create([
    'http' => [
        'method' => 'GET',
        'header' => 'Content-Type: application/json'
    ]
]);

$response = file_get_contents($apiUrl, false, $context);
if ($response) {
    $data = json_decode($response, true);
    if ($data && isset($data['success']) && $data['success']) {
        $deviceCount = count($data['data']);
        echo "✅ API endpoint working! Found $deviceCount devices.\n";
        
        // Check if our test device is in the API response
        $found = false;
        foreach ($data['data'] as $device) {
            if ($device['device_id'] === $testDeviceId) {
                $found = true;
                break;
            }
        }
        
        if ($found) {
            echo "✅ Test device found in API response!\n";
        } else {
            echo "❌ Test device not found in API response!\n";
        }
    } else {
        echo "❌ API endpoint returned error: " . $response . "\n";
    }
} else {
    echo "❌ API endpoint not accessible!\n";
}

// Test 11: Cleanup
echo "\n🧹 Cleaning up test data...\n";
try {
    // Delete test device and related data
    $pdo->prepare("DELETE FROM locations WHERE device_id = ?")->execute([$testDeviceId]);
    $pdo->prepare("DELETE FROM devices WHERE device_id = ?")->execute([$testDeviceId]);
    echo "✅ Test data cleaned up successfully!\n";
} catch (Exception $e) {
    echo "⚠️ Cleanup warning: " . $e->getMessage() . "\n";
}

echo "\n🎉 End-to-End Device Registration Test Completed!\n";
echo "==============================================\n";
echo "If all tests passed, your monitoring system is working correctly!\n";
echo "The Android app should now be able to register devices and upload data.\n";
?>
