<?php
// HTTP Client Test - Simulates Android app HTTP requests
// This tests the actual HTTP endpoints that the Android app uses

echo "🌐 Testing HTTP Client (Android App Simulation)\n";
echo "===============================================\n\n";

// Configuration
$baseUrl = "http://mobiletr.upastithi.in/php/";
$testDeviceId = "http_test_device_" . time();
$timestamp = time() * 1000;
$apiKey = "monitoring_key_2024";
$encryptionKey = "monitoring_secret_key_2024";

// Simulate Android encryption
function androidEncrypt($data, $key) {
    $result = '';
    for ($i = 0; $i < strlen($data); $i++) {
        $result .= chr(ord($data[$i]) ^ ord($key[$i % strlen($key)]));
    }
    return $result;
}

// Test 1: Test Device Registration via Location Upload
echo "1️⃣ Testing Device Registration via Location Upload...\n";

$locationData = [
    'latitude' => 28.6139,
    'longitude' => 77.2090,
    'accuracy' => 10.5,
    'timestamp' => $timestamp,
    'address' => 'New Delhi, India'
];

$encryptedData = androidEncrypt(json_encode($locationData), $encryptionKey);
$authToken = androidEncrypt("$testDeviceId:$timestamp:$apiKey", $encryptionKey);

// Prepare headers
$headers = [
    'X-Device-ID: ' . $testDeviceId,
    'X-Timestamp: ' . $timestamp,
    'X-Auth-Token: ' . $authToken,
    'X-Data-Type: location',
    'Content-Type: application/json'
];

// Make HTTP request
$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, $baseUrl . 'upload.php');
curl_setopt($ch, CURLOPT_POST, true);
curl_setopt($ch, CURLOPT_POSTFIELDS, $encryptedData);
curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_TIMEOUT, 30);

$response = curl_exec($ch);
$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
$error = curl_error($ch);
curl_close($ch);

if ($error) {
    echo "❌ cURL Error: $error\n";
} else {
    echo "HTTP Status Code: $httpCode\n";
    echo "Response: $response\n";
    
    if ($httpCode === 200) {
        $data = json_decode($response, true);
        if ($data && isset($data['success']) && $data['success']) {
            echo "✅ Location upload successful! Device should be registered.\n";
        } else {
            echo "❌ Location upload failed: " . ($data['message'] ?? 'Unknown error') . "\n";
        }
    } else {
        echo "❌ HTTP request failed with status $httpCode\n";
    }
}

// Test 2: Test API Endpoint to Verify Device
echo "\n2️⃣ Testing API Endpoint to Verify Device...\n";

$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, $baseUrl . 'api.php?action=devices');
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_TIMEOUT, 30);

$response = curl_exec($ch);
$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
$error = curl_error($ch);
curl_close($ch);

if ($error) {
    echo "❌ cURL Error: $error\n";
} else {
    echo "HTTP Status Code: $httpCode\n";
    echo "Response: $response\n";
    
    if ($httpCode === 200) {
        $data = json_decode($response, true);
        if ($data && isset($data['success']) && $data['success']) {
            $devices = $data['data'];
            echo "✅ API endpoint working! Found " . count($devices) . " devices.\n";
            
            // Check if our test device is in the list
            $found = false;
            foreach ($devices as $device) {
                if ($device['device_id'] === $testDeviceId) {
                    $found = true;
                    echo "✅ Test device found in API response!\n";
                    echo "  Device ID: " . $device['device_id'] . "\n";
                    echo "  Device Name: " . $device['device_name'] . "\n";
                    echo "  Last Seen: " . $device['last_seen'] . "\n";
                    break;
                }
            }
            
            if (!$found) {
                echo "❌ Test device not found in API response!\n";
                echo "Available devices:\n";
                foreach ($devices as $device) {
                    echo "  - " . $device['device_id'] . " (" . $device['device_name'] . ")\n";
                }
            }
        } else {
            echo "❌ API returned error: " . ($data['message'] ?? 'Unknown error') . "\n";
        }
    } else {
        echo "❌ HTTP request failed with status $httpCode\n";
    }
}

// Test 3: Test Screenshot Upload
echo "\n3️⃣ Testing Screenshot Upload...\n";

// Create a test image file
$testImagePath = '/tmp/test_screenshot.png';
$image = imagecreate(100, 100);
$bgColor = imagecolorallocate($image, 255, 255, 255);
$textColor = imagecolorallocate($image, 0, 0, 0);
imagestring($image, 5, 20, 40, 'TEST', $textColor);
imagepng($image, $testImagePath);
imagedestroy($image);

// Prepare multipart form data
$boundary = '----WebKitFormBoundary' . uniqid();
$data = '';

// Add file
$data .= "--$boundary\r\n";
$data .= "Content-Disposition: form-data; name=\"file\"; filename=\"test_screenshot.png\"\r\n";
$data .= "Content-Type: image/png\r\n\r\n";
$data .= file_get_contents($testImagePath) . "\r\n";

// Add data type
$data .= "--$boundary\r\n";
$data .= "Content-Disposition: form-data; name=\"data_type\"\r\n\r\n";
$data .= "screenshot\r\n";

$data .= "--$boundary--\r\n";

// Prepare headers
$headers = [
    'X-Device-ID: ' . $testDeviceId,
    'X-Timestamp: ' . $timestamp,
    'X-Auth-Token: ' . $authToken,
    'Content-Type: multipart/form-data; boundary=' . $boundary
];

// Make HTTP request
$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, $baseUrl . 'upload.php');
curl_setopt($ch, CURLOPT_POST, true);
curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_TIMEOUT, 30);

$response = curl_exec($ch);
$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
$error = curl_error($ch);
curl_close($ch);

// Clean up test image
unlink($testImagePath);

if ($error) {
    echo "❌ cURL Error: $error\n";
} else {
    echo "HTTP Status Code: $httpCode\n";
    echo "Response: $response\n";
    
    if ($httpCode === 200) {
        $data = json_decode($response, true);
        if ($data && isset($data['success']) && $data['success']) {
            echo "✅ Screenshot upload successful!\n";
        } else {
            echo "❌ Screenshot upload failed: " . ($data['message'] ?? 'Unknown error') . "\n";
        }
    } else {
        echo "❌ HTTP request failed with status $httpCode\n";
    }
}

echo "\n🎉 HTTP Client Test Completed!\n";
echo "=============================\n";
echo "If all tests passed, your Android app should work correctly!\n";
echo "Check your dashboard at: http://mobiletr.upastithi.in/dashboard/\n";
?>
