<?php
// Real API endpoint testing - simulates actual Android app requests
echo "🌐 Testing Real API Endpoints\n";
echo "============================\n\n";

$baseUrl = "http://mobiletr.upastithi.in/php/";
$dashboardUrl = "http://mobiletr.upastithi.in/dashboard/";

// Test 1: Test Dashboard Accessibility
echo "1️⃣ Testing Dashboard Accessibility...\n";
$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, $dashboardUrl);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
curl_setopt($ch, CURLOPT_TIMEOUT, 30);

$response = curl_exec($ch);
$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
$error = curl_error($ch);
curl_close($ch);

if ($error) {
    echo "❌ Dashboard Error: $error\n";
} else {
    echo "✅ Dashboard HTTP Status: $httpCode\n";
    if ($httpCode === 200) {
        echo "✅ Dashboard is accessible!\n";
        if (strpos($response, 'loginModal') !== false) {
            echo "✅ Login modal found in dashboard\n";
        } else {
            echo "❌ Login modal not found\n";
        }
    } else {
        echo "❌ Dashboard not accessible (HTTP $httpCode)\n";
    }
}

// Test 2: Test Authentication
echo "\n2️⃣ Testing Dashboard Authentication...\n";
$authUrl = $baseUrl . "auth.php";
$password = "admin123";

$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, $authUrl);
curl_setopt($ch, CURLOPT_POST, true);
curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode(['password' => $password]));
curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_COOKIEJAR, '/tmp/test_cookies.txt');
curl_setopt($ch, CURLOPT_COOKIEFILE, '/tmp/test_cookies.txt');
curl_setopt($ch, CURLOPT_TIMEOUT, 30);

$response = curl_exec($ch);
$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
$error = curl_error($ch);

if ($error) {
    echo "❌ Auth Error: $error\n";
} else {
    echo "Auth HTTP Status: $httpCode\n";
    echo "Auth Response: $response\n";
    
    if ($httpCode === 200) {
        $data = json_decode($response, true);
        if ($data && isset($data['success']) && $data['success']) {
            echo "✅ Authentication successful!\n";
        } else {
            echo "❌ Authentication failed: " . ($data['message'] ?? 'Unknown error') . "\n";
        }
    } else {
        echo "❌ Authentication request failed (HTTP $httpCode)\n";
    }
}

// Test 3: Test API Endpoint (Devices) with Authentication
echo "\n3️⃣ Testing API Endpoint (Devices) with Authentication...\n";
$apiUrl = $baseUrl . "api.php?action=devices";

curl_setopt($ch, CURLOPT_URL, $apiUrl);
curl_setopt($ch, CURLOPT_POST, false);
curl_setopt($ch, CURLOPT_HTTPGET, true);
curl_setopt($ch, CURLOPT_HTTPHEADER, [
    'Content-Type: application/json',
    'X-Requested-With: XMLHttpRequest'
]);

$response = curl_exec($ch);
$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
$error = curl_error($ch);

if ($error) {
    echo "❌ API Error: $error\n";
} else {
    echo "API HTTP Status: $httpCode\n";
    echo "API Response: $response\n";
    
    if ($httpCode === 200) {
        $data = json_decode($response, true);
        if ($data && isset($data['success']) && $data['success']) {
            $devices = $data['data'];
            echo "✅ API working! Found " . count($devices) . " devices.\n";
            
            if (!empty($devices)) {
                echo "Devices in database:\n";
                foreach ($devices as $device) {
                    echo "  📱 Device ID: " . $device['device_id'] . "\n";
                    echo "     Name: " . $device['device_name'] . "\n";
                    echo "     Last Seen: " . $device['last_seen'] . "\n";
                    echo "     Calls: " . $device['call_count'] . "\n";
                    echo "     Locations: " . $device['location_count'] . "\n";
                    echo "     Media: " . $device['media_count'] . "\n";
                    echo "     ---\n";
                }
            } else {
                echo "ℹ️ No devices found in database yet.\n";
            }
        } else {
            echo "❌ API returned error: " . ($data['message'] ?? 'Unknown error') . "\n";
        }
    } elseif ($httpCode === 401) {
        echo "❌ API requires authentication (401 Unauthorized)\n";
    } else {
        echo "❌ API request failed (HTTP $httpCode)\n";
    }
}

// Test 4: Simulate Android App Device Registration
echo "\n4️⃣ Simulating Android App Device Registration...\n";
$testDeviceId = "real_test_device_" . time();
$timestamp = time() * 1000;
$apiKey = "monitoring_key_2024";
$encryptionKey = "monitoring_secret_key_2024";

// Simulate Android encryption
function androidEncrypt($data, $key) {
    $result = '';
    for ($i = 0; $i < strlen($data); $i++) {
        $result .= chr(ord($data[$i]) ^ ord($key[$i % strlen($key)]));
    }
    return $result;
}

$locationData = [
    'latitude' => 28.6139,
    'longitude' => 77.2090,
    'accuracy' => 10.5,
    'timestamp' => $timestamp,
    'address' => 'New Delhi, India'
];

$encryptedData = androidEncrypt(json_encode($locationData), $encryptionKey);
$authToken = androidEncrypt("$testDeviceId:$timestamp:$apiKey", $encryptionKey);

// Test device registration via upload.php
$uploadUrl = $baseUrl . "upload.php";
$headers = [
    'X-Device-ID: ' . $testDeviceId,
    'X-Timestamp: ' . $timestamp,
    'X-Auth-Token: ' . $authToken,
    'X-Data-Type: location',
    'Content-Type: application/json'
];

curl_setopt($ch, CURLOPT_URL, $uploadUrl);
curl_setopt($ch, CURLOPT_POST, true);
curl_setopt($ch, CURLOPT_POSTFIELDS, $encryptedData);
curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);

$response = curl_exec($ch);
$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
$error = curl_error($ch);

if ($error) {
    echo "❌ Upload Error: $error\n";
} else {
    echo "Upload HTTP Status: $httpCode\n";
    echo "Upload Response: $response\n";
    
    if ($httpCode === 200) {
        $data = json_decode($response, true);
        if ($data && isset($data['success']) && $data['success']) {
            echo "✅ Device registration successful!\n";
        } else {
            echo "❌ Device registration failed: " . ($data['message'] ?? 'Unknown error') . "\n";
        }
    } else {
        echo "❌ Upload request failed (HTTP $httpCode)\n";
    }
}

// Test 5: Verify Device Appears in API
echo "\n5️⃣ Verifying Device Appears in API...\n";
curl_setopt($ch, CURLOPT_URL, $apiUrl);
curl_setopt($ch, CURLOPT_POST, false);
curl_setopt($ch, CURLOPT_HTTPGET, true);
curl_setopt($ch, CURLOPT_HTTPHEADER, [
    'Content-Type: application/json',
    'X-Requested-With: XMLHttpRequest'
]);

$response = curl_exec($ch);
$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
$error = curl_error($ch);
curl_close($ch);

if ($error) {
    echo "❌ Verification Error: $error\n";
} else {
    echo "Verification HTTP Status: $httpCode\n";
    
    if ($httpCode === 200) {
        $data = json_decode($response, true);
        if ($data && isset($data['success']) && $data['success']) {
            $devices = $data['data'];
            echo "✅ API working! Found " . count($devices) . " devices.\n";
            
            $found = false;
            foreach ($devices as $device) {
                if ($device['device_id'] === $testDeviceId) {
                    $found = true;
                    echo "✅ Test device found in API response!\n";
                    echo "  📱 Device ID: " . $device['device_id'] . "\n";
                    echo "     Name: " . $device['device_name'] . "\n";
                    echo "     Last Seen: " . $device['last_seen'] . "\n";
                    break;
                }
            }
            
            if (!$found) {
                echo "❌ Test device not found in API response!\n";
                echo "Available devices:\n";
                foreach ($devices as $device) {
                    echo "  - " . $device['device_id'] . " (" . $device['device_name'] . ")\n";
                }
            }
        } else {
            echo "❌ API verification failed: " . ($data['message'] ?? 'Unknown error') . "\n";
        }
    } else {
        echo "❌ API verification request failed (HTTP $httpCode)\n";
    }
}

// Test 6: Test Dashboard JavaScript API Calls
echo "\n6️⃣ Testing Dashboard JavaScript API Calls...\n";
$jsTestUrl = $baseUrl . "test_dashboard_js.php";
$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, $jsTestUrl);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_TIMEOUT, 30);

$response = curl_exec($ch);
$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
$error = curl_error($ch);
curl_close($ch);

if ($error) {
    echo "❌ JS Test Error: $error\n";
} else {
    echo "JS Test HTTP Status: $httpCode\n";
    if ($httpCode === 200) {
        echo "✅ Dashboard JavaScript test completed\n";
        echo "Response: $response\n";
    } else {
        echo "❌ Dashboard JavaScript test failed (HTTP $httpCode)\n";
    }
}

// Cleanup
unlink('/tmp/test_cookies.txt');

echo "\n🎉 Real API Testing Completed!\n";
echo "=============================\n";
echo "Summary:\n";
echo "- Dashboard URL: $dashboardUrl\n";
echo "- API Base URL: $baseUrl\n";
echo "- Default Password: admin123\n";
echo "\nTo access your dashboard:\n";
echo "1. Go to: $dashboardUrl\n";
echo "2. Enter password: admin123\n";
echo "3. Click Login\n";
echo "4. You should see your devices!\n";
?>
