<?php
// Direct upload test - bypasses curl issues
echo "🔧 Direct Upload Test\n";
echo "====================\n\n";

// Simulate the upload.php processing directly
require_once 'php/config/database.php';

// Test device data
$testDeviceId = "direct_test_" . time();
$timestamp = time() * 1000;
$apiKey = "monitoring_key_2024";
$encryptionKey = "monitoring_secret_key_2024";

// Simulate Android encryption
function androidEncrypt($data, $key) {
    $result = '';
    for ($i = 0; $i < strlen($data); $i++) {
        $result .= chr(ord($data[$i]) ^ ord($key[$i % strlen($key)]));
    }
    return $result;
}

$locationData = [
    'latitude' => 28.6139,
    'longitude' => 77.2090,
    'accuracy' => 10.5,
    'timestamp' => $timestamp,
    'address' => 'New Delhi, India'
];

$encryptedData = androidEncrypt(json_encode($locationData), $encryptionKey);
$authToken = androidEncrypt("$testDeviceId:$timestamp:$apiKey", $encryptionKey);

echo "Test Device ID: $testDeviceId\n";
echo "Timestamp: $timestamp\n";
echo "Auth Token (first 20 chars): " . substr($authToken, 0, 20) . "...\n\n";

// Test 1: Test authentication validation
echo "1️⃣ Testing Authentication Validation...\n";
$headers = [
    'X-Device-ID' => $testDeviceId,
    'X-Timestamp' => $timestamp,
    'X-Auth-Token' => $authToken,
    'X-Data-Type' => 'location'
];

$isValid = validateDeviceAuth($headers);
if ($isValid) {
    echo "✅ Authentication validation successful!\n";
} else {
    echo "❌ Authentication validation failed!\n";
    echo "This means the encryption keys don't match between Android and server.\n";
    echo "Expected token: " . substr($authToken, 0, 20) . "...\n";
    
    // Let's check what the server expects
    $expectedToken = encryptData("$testDeviceId:$timestamp:" . API_KEY, ENCRYPTION_KEY);
    echo "Server expects: " . substr($expectedToken, 0, 20) . "...\n";
    echo "Keys match: " . (hash_equals($expectedToken, $authToken) ? 'Yes' : 'No') . "\n";
}

// Test 2: Test database connection
echo "\n2️⃣ Testing Database Connection...\n";
$pdo = getDatabaseConnection();
if (!$pdo) {
    echo "❌ Database connection failed!\n";
    exit(1);
}
echo "✅ Database connection successful!\n";

// Test 3: Test device registration directly
echo "\n3️⃣ Testing Device Registration Directly...\n";
try {
    $stmt = $pdo->prepare("
        INSERT INTO devices (device_id, device_name) 
        VALUES (?, ?) 
        ON DUPLICATE KEY UPDATE last_seen = CURRENT_TIMESTAMP
    ");
    $result = $stmt->execute([$testDeviceId, "Direct Test Device"]);
    
    if ($result) {
        echo "✅ Device registered successfully in database!\n";
    } else {
        echo "❌ Device registration failed!\n";
    }
} catch (Exception $e) {
    echo "❌ Device registration error: " . $e->getMessage() . "\n";
}

// Test 4: Test location data insertion
echo "\n4️⃣ Testing Location Data Insertion...\n";
try {
    $stmt = $pdo->prepare("
        INSERT INTO locations (device_id, latitude, longitude, accuracy, timestamp, address) 
        VALUES (?, ?, ?, ?, ?, ?)
    ");
    
    $result = $stmt->execute([
        $testDeviceId,
        $locationData['latitude'],
        $locationData['longitude'],
        $locationData['accuracy'],
        $locationData['timestamp'],
        $locationData['address']
    ]);
    
    if ($result) {
        echo "✅ Location data inserted successfully!\n";
    } else {
        echo "❌ Location data insertion failed!\n";
    }
} catch (Exception $e) {
    echo "❌ Location data insertion error: " . $e->getMessage() . "\n";
}

// Test 5: Verify data in database
echo "\n5️⃣ Verifying Data in Database...\n";
try {
    $stmt = $pdo->prepare("SELECT * FROM devices WHERE device_id = ?");
    $stmt->execute([$testDeviceId]);
    $device = $stmt->fetch();
    
    if ($device) {
        echo "✅ Device found in database!\n";
        echo "  Device ID: " . $device['device_id'] . "\n";
        echo "  Name: " . $device['device_name'] . "\n";
        echo "  Last Seen: " . $device['last_seen'] . "\n";
    } else {
        echo "❌ Device not found in database!\n";
    }
    
    $stmt = $pdo->prepare("SELECT * FROM locations WHERE device_id = ?");
    $stmt->execute([$testDeviceId]);
    $location = $stmt->fetch();
    
    if ($location) {
        echo "✅ Location data found in database!\n";
        echo "  Location: " . $location['latitude'] . ", " . $location['longitude'] . "\n";
        echo "  Address: " . $location['address'] . "\n";
    } else {
        echo "❌ Location data not found in database!\n";
    }
} catch (Exception $e) {
    echo "❌ Database verification error: " . $e->getMessage() . "\n";
}

// Test 6: Test public API
echo "\n6️⃣ Testing Public API...\n";
$devicesUrl = "http://mobiletr.upastithi.in/php/public_api.php?action=devices";

$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, $devicesUrl);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_TIMEOUT, 30);

$response = curl_exec($ch);
$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
$error = curl_error($ch);
curl_close($ch);

if ($error) {
    echo "❌ cURL Error: $error\n";
} else {
    echo "HTTP Status: $httpCode\n";
    echo "Response: $response\n";
    
    if ($httpCode === 200) {
        $data = json_decode($response, true);
        if ($data && isset($data['success']) && $data['success']) {
            $devices = $data['data'];
            echo "✅ Public API working! Found " . count($devices) . " devices.\n";
            
            $found = false;
            foreach ($devices as $device) {
                if ($device['device_id'] === $testDeviceId) {
                    $found = true;
                    echo "✅ Test device found in public API!\n";
                    break;
                }
            }
            
            if (!$found) {
                echo "❌ Test device not found in public API.\n";
            }
        } else {
            echo "❌ Public API error: " . ($data['message'] ?? 'Unknown error') . "\n";
        }
    } else {
        echo "❌ Public API failed (HTTP $httpCode)\n";
    }
}

// Cleanup
echo "\n7️⃣ Cleaning up test data...\n";
try {
    $pdo->prepare("DELETE FROM locations WHERE device_id = ?")->execute([$testDeviceId]);
    $pdo->prepare("DELETE FROM devices WHERE device_id = ?")->execute([$testDeviceId]);
    echo "✅ Test data cleaned up.\n";
} catch (Exception $e) {
    echo "⚠️ Cleanup warning: " . $e->getMessage() . "\n";
}

echo "\n✅ Direct Upload Test Completed!\n";
echo "===============================\n";
echo "This test bypasses the HTTP upload and tests the core functionality directly.\n";
echo "If this works but HTTP upload doesn't, the issue is with the web server configuration.\n";
?>
